/*
**
** This is the assembler part of the CD-XFS for MagiC
** Developed with PASM.
**
** (C) Andreas Kromke 1997
**
**
*/

     INCLUDE "MGX_XFS.INC"


/*
*
* This is the passing structure between the
* assembler part of the XFS and the C part.
* The C part gets its parameters via the
* stack ("cdecl"-convention), so that any
* desired compilers may be used.
*
*/

     XREF cdecl_cdxfs              ; XFS jump-table
     XREF cdecl_cddev              ; File driver


/*
*
* These are the actual drivers, which communicate 
* via an assembler interface with the MagiC kernel 
* and via a C interface with the MAC part.
*
*/

     XDEF cdxfs                    ; Here is the XFS
     XDEF cddev                    ; Here the file driver


cdxfs:
 DC.B     'CDROM',0,0,0            ; Name
 DC.L     0                        ; Next XFS
 DC.L     0                        ; Flags
 DC.L     0                        ; xfs_init
 DC.L     cdxfs_sync
 DC.L     cdxfs_pterm
 DC.L     cdxfs_garbcoll
 DC.L     cdxfs_freeDD
 DC.L     cdxfs_drv_open
 DC.L     cdxfs_drv_close
 DC.L     cdxfs_path2DD
 DC.L     cdxfs_sfirst
 DC.L     cdxfs_snext
 DC.L     cdxfs_fopen
 DC.L     cdxfs_fdelete
 DC.L     cdxfs_link
 DC.L     cdxfs_xattr
 DC.L     cdxfs_attrib
 DC.L     cdxfs_chown
 DC.L     cdxfs_chmod
 DC.L     cdxfs_dcreate
 DC.L     cdxfs_ddelete
 DC.L     cdxfs_DD2name
 DC.L     cdxfs_dopendir
 DC.L     cdxfs_dreaddir
 DC.L     cdxfs_drewinddir
 DC.L     cdxfs_dclosedir
 DC.L     cdxfs_dpathconf
 DC.L     cdxfs_dfree
 DC.L     cdxfs_wlabel
 DC.L     cdxfs_rlabel
 DC.L     cdxfs_symlink
 DC.L     cdxfs_readlink
 DC.L     cdxfs_dcntl


cddev:
 DC.L     cddev_close
 DC.L     cddev_read
 DC.L     cddev_write
 DC.L     cddev_stat
 DC.L     cddev_seek
 DC.L     cddev_datime
 DC.L     cddev_ioctl
 DC.L     cddev_getc
 DC.L     cddev_getline
 DC.L     cddev_putc



**********************************************************************
*
* void xfs_sync( a0 = DMD *d )
*

cdxfs_sync:
 move.l   a0,-(sp)
 move.l   cdecl_cdxfs+xfs_sync,a0
 jsr      (a0)
 addq.l   #4,sp
 rts


**********************************************************************
*
* void xfs_pterm( a0 = DMD *d, a1 = PD *pd )
*
* A program has just been terminated. The XFS can free all the 
* resources occupied by the program.
* All resources that the kernel knows about (i.e. opened files)
* have been freed already by the kernel.
*

cdxfs_pterm:
 move.l   a1,-(sp)
 move.l   a0,-(sp)
 move.l   cdecl_cdxfs+xfs_pterm,a0
 jsr      (a0)
 addq.l   #8,sp
 rts


**********************************************************************
*
* long xfs_garbcoll( a0 = DMD *dir )
*
* Searches for an unused FD
* Returns TRUE if at least one was found.
*

cdxfs_garbcoll:
 move.l   a0,-(sp)
 move.l   cdecl_cdxfs+xfs_garbcoll,a0
 jsr      (a0)
 addq.l   #4,sp
 rts


**********************************************************************
*
* void xfs_freeDD( a0 = DD *dir )
*
* The kernel has decremented the reference counter of the DD to 0.
* The structure can now be freed.
*

cdxfs_freeDD:
 move.l   a0,-(sp)
 move.l   cdecl_cdxfs+xfs_freeDD,a0
 jsr      (a0)
 addq.l   #4,sp
 rts


**********************************************************************
*
* long xfs_drv_open( a0 = DMD *dmd )
*
* Initialises the DMD.
* Disk changes on the MAC-side are not possible at present,
* hence an E_OK is returned here already.
*

cdxfs_drv_open:
 move.l   a0,-(sp)
 move.l   cdecl_cdxfs+xfs_drv_open,a0
 jsr      (a0)
 addq.l   #4,sp
 rts


**********************************************************************
*
* long xfs_drv_close( a0 = DMD *dmd, d0 = int mode)
*
* mode == 0:   Ask if closing allowed, close if appropriate
*         1:   Force closing, must return E_OK
*

cdxfs_drv_close:
 move.l   a0,-(sp)
 move.w   d0,-(sp)
 move.l   cdecl_cdxfs+xfs_drv_close,a0
 jsr      (a0)
 addq.l   #6,sp
 rts


**********************************************************************
*
* DD * xfs_path2DD( a0 = DD * reldir,
*                    a1 = char *pathname,
*                    d0 = int  mode )
*
* d1 is the path without drive letter
* -> d0 = DD *dir     Or error code
* -> d1 = char *fname
*
* Converts the pathname that is specified relative to <reldir>
* into a DD.
*
* mode == 0: pathname points to any desired file; return the DD
*            that contains the file.
*            In a0 return a pointer to the isolated filename.
*         1: pathname is itself a directory, return its DD;
*            a0 is undefined after this.
*
* Returns:
*  d0 = DD of the path, the reference counter is incremented accordingly
*  d1 = Rest of filename without leading '\'
* or
*  d0 = ELINK
*  d1 = Rest of path without leading '\'
*  a0 = DD of the path that contains the symbolic link. This is 
*       important for relative path specifications in the link.
*  a1 = NULL
*            The path represents the parent of the root directory,
*            the kernel can, if the drive is U:, go back to U:\.
*  a1 = Path of the symbolic link. The path contains a symbolic
*            link, if possible to a different drive/partition.
*            The kernel has to convert the rest path <a0> relative
*            to the new DD <a0>.
*            a1 points to a word for the character string length
*            (even number at an even address, incl. EOS), which 
*            is followed by the character string. The buffer can 
*            be fugitive as the kernel copies the path.
*
*
* At present no symlinks are suported. The parent of a root directory
* is also not handled correctly.
* It would be sensible to have a overview of all requested DDs, so that
* an already referenced dirID does not have to request a new descriptor,
* but simply to increment the reference counter.
*

cdxfs_path2DD:
 clr.l    -(sp)                    ; Space for return symlink
 clr.l    -(sp)                    ; Space for return symlink-DD
 clr.l    -(sp)                    ; Space for return restpath
 pea      8(sp)                    ; &symlink
 pea      8(sp)                    ; &dd
 pea      8(sp)                    ; &restpath
 move.w   d0,-(sp)                 ; mode
 move.l   a1,-(sp)                 ; pathname
 move.l   a0,-(sp)                 ; reldir
 move.l   cdecl_cdxfs+xfs_path2DD,a0
 jsr      (a0)
 adda.w   #22,sp
 move.l   (sp)+,d1                  ; restpath
 move.l   (sp)+,a0                  ; Symlink-DD
 move.l   (sp)+,a1                  ; Symlink
 rts


**********************************************************************
*
* long xfs_sfirst(a0 = DD *d, a1 = char *name, d0 = DTA *dta,
*                  d1 = int attrib)
*
* Returns:     d0 = errcode
*             or
*              d0 = ELINK
*              a0 = char *link
*

cdxfs_sfirst:
 clr.l    -(sp)
 pea      (sp)
 move.w   d1,-(sp)
 move.l   d0,-(sp)
 move.l   a1,-(sp)
 move.l   a0,-(sp)
 move.l   cdecl_cdxfs+xfs_sfirst,a0
 jsr      (a0)
 adda.w   #18,sp
 move.l   (sp)+,a0
 rts


**********************************************************************
*
* long xfs_snext(a0 = DTA *dta, a1 = DMD *d)
*
* Returns:     d0 = errcode
*             or
*              d0 = ELINK
*              a0 = char *link
*

cdxfs_snext:
 clr.l    -(sp)
 pea      (sp)
 move.l   a1,-(sp)
 move.l   a0,-(sp)
 move.l   cdecl_cdxfs+xfs_snext,a0
 jsr      (a0)
 adda.w   #12,sp
 move.l   (sp)+,a0
 rts


**********************************************************************
*
* d0 = FD * xfs_fopen(a0 = DD *d, a1 = char *name, d0 = int omode,
*                      d1 = int attrib )
*
* Opens and/or creates files, opens the file driver.
* The Open mode has already been converted to the internal
* MagiC specification by the kernel.
*
* A repetition in case of E_CHNG will be undertaken by the kernel.
*
* Returns:
* d0 = ELINK: File is a symbolic link
*             a0 is the filename of the symbolic link
*

cdxfs_fopen:
 clr.l    -(sp)
 pea      (sp)
 move.w   d1,-(sp)
 move.l   d0,-(sp)
 move.l   a1,-(sp)
 move.l   a0,-(sp)
 move.l   cdecl_cdxfs+xfs_fopen,a0
 jsr      (a0)
 adda.w   #18,sp
 move.l   (sp)+,a0
 rts


*********************************************************************
*
* long xfs_fdelete(a0 = DD *d, a1 = char *name)
*
* A repetition in case of E_CHNG will be undertaken by the kernel.
*
* Returns:
* d0 = ELINK: File is a symbolic link
*             a0 is the filename of the symbolic link
*
* No subdirs or labels may be deleted.
*

cdxfs_fdelete:
 move.l   a1,-(sp)
 move.l   a0,-(sp)
 move.l   cdecl_cdxfs+xfs_fdelete,a0
 jsr      (a0)
 addq.l   #8,sp
 rts


**********************************************************************
*
* long xfs_link(a0 = DD *olddir, a1 = DD *newdir,
*                   d0 = char *oldname, d1 = char *newname,
*                   d2 = int flag)
*
* d2 = 1: Flink
* d2 = 0: Frename
*

cdxfs_link:
 move.w   d2,-(sp)
 move.l   d1,-(sp)
 move.l   d0,-(sp)
 move.l   a1,-(sp)
 move.l   a0,-(sp)
 move.l   cdecl_cdxfs+xfs_link,a0
 jsr      (a0)
 adda.w   #18,sp
 rts


**********************************************************************
*
* long xfs_xattr( a0 = DD *dir, a1 = char *name, d0 = XATTR *xa,
*                  d1 = int mode )
*
* mode == 0:   Follow symbolic links  (i.e. return ELINK)
*         1:   Don't follow (i.e. create XATTR for the link)
*

cdxfs_xattr:
 move.l   d1,-(sp)
 move.l   d0,-(sp)
 move.l   a1,-(sp)
 move.l   a0,-(sp)
 move.l   cdecl_cdxfs+xfs_xattr,a0
 jsr      (a0)
 adda.w   #16,sp
 rts


**********************************************************************
*
* long xfs_attrib( a0 = DD *dir, a1 = char *name, d0 = int mode,
*                   d1 = int attrib )
*
* Returns:     >= 0      Attribute
*              <  0      Error 
*
* mode == 0:   Read attribute
*         1:   Write attribute
*

cdxfs_attrib:
 move.w   d1,-(sp)
 move.w   d0,-(sp)
 move.l   a1,-(sp)
 move.l   a0,-(sp)
 move.l   cdecl_cdxfs+xfs_attrib,a0
 jsr      (a0)
 adda.w   #12,sp
 rts


**********************************************************************
*
* long xfs_chown( a0 = DD *dir, a1 = char *name, d0 = int uid,
*                  d1 = int gid )
*
* Returns :    == 0      OK
*              <  0      Error
*

cdxfs_chown:
 move.w   d1,-(sp)
 move.w   d0,-(sp)
 move.l   a1,-(sp)
 move.l   a0,-(sp)
 move.l   cdecl_cdxfs+xfs_chown,a0
 jsr      (a0)
 adda.w   #12,sp
 rts


**********************************************************************
*
* long xfs_chmod( a0 = DD *dir, a1 = char *name, d0 = int mode )
*
* Returns:     == 0      OK
*              <  0      Error
*

cdxfs_chmod:
 move.w   d0,-(sp)
 move.l   a1,-(sp)
 move.l   a0,-(sp)
 move.l   cdecl_cdxfs+xfs_chmod,a0
 jsr      (a0)
 adda.w   #10,sp
 rts


**********************************************************************
*
* long xfs_dcreate(a0 = DD *d, a1 = char *name, d0 = int mode )
*
* mode is ignored here
*

cdxfs_dcreate:
 move.l   a1,-(sp)
 move.l   a0,-(sp)
 move.l   cdecl_cdxfs+xfs_dcreate,a0
 jsr      (a0)
 addq.l   #8,sp
 rts


**********************************************************************
*
* long xfs_ddelete( a0 = DD *d )
*
* The DD may not be released (remains locked!)
*

cdxfs_ddelete:
 move.l   a0,-(sp)
 move.l   cdecl_cdxfs+xfs_ddelete,a0
 jsr      (a0)
 addq.l   #4,sp
 rts


**********************************************************************
*
* long xfs_DD2name(a0 = DD *d, a1 = char *buf, d0 = int buflen)
*
* Converts DD to a pathname
*

cdxfs_DD2name:
 move.w   d0,-(sp)
 move.l   a1,-(sp)
 move.l   a0,-(sp)
 move.l   cdecl_cdxfs+xfs_DD2name,a0
 jsr      (a0)
 adda.w   #10,sp
 rts


**********************************************************************
*
* FD *xfs_dopendir( a0 = DD *d, d0 = int tosflag )
*

cdxfs_dopendir:
 move.w   d0,-(sp)
 move.l   a0,-(sp)
 move.l   cdecl_cdxfs+xfs_dopendir,a0
 jsr      (a0)
 addq.l   #6,sp
 rts


**********************************************************************
*
* long xfs_dreaddir( a0 = void *dh, d0 = int len, a1 = char *buf,
*                     d1 = XATTR *xattr, d2 = long *xr )
*
* For Dreaddir (xattr = NULL) and Dxreaddir
*

cdxfs_dreaddir:
 move.l   d2,-(sp)
 move.l   d1,-(sp)
 move.l   a1,-(sp)
 move.w   d0,-(sp)
 move.l   a0,-(sp)
 move.l   cdecl_cdxfs+xfs_dreaddir,a0
 jsr      (a0)
 adda.w   #18,sp
 rts


**********************************************************************
*
* long xfs_drewinddir( a0 = FD *d )
*

cdxfs_drewinddir:
 move.l   a0,-(sp)
 move.l   cdecl_cdxfs+xfs_drewinddir,a0
 jsr      (a0)
 addq.l   #4,sp
 rts


**********************************************************************
*
* long xfs_dclosedir( a0 = FD *d )
*

cdxfs_dclosedir:
 move.l   a0,-(sp)
 move.l   cdecl_cdxfs+xfs_dclosedir,a0
 jsr      (a0)
 addq.l   #4,sp
 rts


**********************************************************************
*
* long xfs_dpathconf( a0 = DD *d, d0 = int which )
*
* mode = -1:   Maximum legal value for n in Dpathconf(n)
*         0:   Internal limit on the number of open files
*         1:   Maximum number of links to a file
*         2:   Maximum length of a full pathname
*         3:   Maximum length of an individual filename
*         4:   Number of bytes that can be written atomically
*         5:   Information about filename truncation
*              0 = Filenames are never truncated; if the filename in
*                  any system call affecting this directory exceeds
*                  the  maximum  length (returned by mode 3), then the
*                  error value ERANGE is returned from that system call.
*
*              1 = Filenames are automatically truncated to the maximum
*                  length.
*
*              2 = Filenames are truncated according to DOS rules,
*                  i.e. to a maximum 8 character base name and a maximum
*                  3 character extension.
*         6:   0 = Case-sensitive
*              1 = Case-insensitive, always in capitals
*              2 = Case-insensitive, but unaffected
*
*      If  any  of these items are unlimited, then 0x7fffffffL is
*      returned.
*

cdxfs_dpathconf:
 move.w   d0,-(sp)
 move.l   a0,-(sp)
 move.l   cdecl_cdxfs+xfs_dpathconf,a0
 jsr      (a0)
 addq.l   #6,sp
 rts


**********************************************************************
*
* long cdxfs_dfree( a0 = DD_FD *dir, a1 = long buf[4] )
*

cdxfs_dfree:
 move.l   a1,-(sp)
 move.l   a0,-(sp)
 move.l   cdecl_cdxfs+xfs_dfree,a0
 jsr      (a0)
 addq.l   #8,sp
 rts


**********************************************************************
*
* long xfs_wlabel( a0 = DD *d, a1 = char *name )
*

cdxfs_wlabel:
 move.l   a1,-(sp)
 move.l   a0,-(sp)
 move.l   cdecl_cdxfs+xfs_wlabel,a0
 jsr      (a0)
 addq.l   #8,sp
 rts


**********************************************************************
*
* long xfs_rlabel( a0 = DD *d, a1 = char *name,
*                   d0 = char *buf, d1 = int len )
*

cdxfs_rlabel:
 move.w   d1,-(sp)
 move.l   d0,-(sp)
 move.l   a1,-(sp)
 move.l   a0,-(sp)
 move.l   cdecl_cdxfs+xfs_rlabel,a0
 jsr      (a0)
 adda.w   #14,sp
 rts


**********************************************************************
*
* long xfs_symlink( a0 = DD *d, a1 = char *name, d0 = char *to )
*
* Create symbolic link
*

cdxfs_symlink:
 move.l   d0,-(sp)
 move.l   a1,-(sp)
 move.l   a0,-(sp)
 move.l   cdecl_cdxfs+xfs_symlink,a0
 jsr      (a0)
 adda.w   #12,sp
 rts


**********************************************************************
*
* long xfs_readlink( a0 = DD *d, a1 = char *name, d0 = char *buf,
*                     d1 = int buflen )
*
* Read symbolic link
*

cdxfs_readlink:
 move.w   d1,-(sp)
 move.l   d0,-(sp)
 move.l   a1,-(sp)
 move.l   a0,-(sp)
 move.l   cdecl_cdxfs+xfs_readlink,a0
 jsr      (a0)
 adda.w   #14,sp
 rts


**********************************************************************
*
* long xfs_dcntl( a0 = DD *d, a1 = char *name, d0 = int cmd,
*                  d1 = long arg )
*
* Executes special functions
*

cdxfs_dcntl:
 move.l   d1,-(sp)
 move.w   d0,-(sp)
 move.l   a1,-(sp)
 move.l   a0,-(sp)
 move.l   cdecl_cdxfs+xfs_dcntl,a0
 jsr      (a0)
 adda.w   #14,sp
 rts



**********************************************************************
**********************************************************************
*
* File driver
*
**********************************************************************
**********************************************************************
*
* long dev_read(a0 = FD *file, d0 = long count, a1 = char *buffer)
*

cddev_read:
 move.l   a1,-(sp)
 move.l   d0,-(sp)
 move.l   a0,-(sp)
 move.l   cdecl_cddev+dev_read,a0
 jsr      (a0)
 adda.w   #12,sp
 rts


**********************************************************************
*
* long dev_write(a0 = FD *file, d0 = long count, a1 = char *buffer)
*

cddev_write:
 move.l   a1,-(sp)
 move.l   d0,-(sp)
 move.l   a0,-(sp)
 move.l   cdecl_cddev+dev_write,a0
 jsr      (a0)
 adda.w   #12,sp
 rts


**********************************************************************
*
* long dev_getc( a0 = FD *f, d0 = int mode )
*
* mode & 0x0001:    cooked
* mode & 0x0002:    echo mode
*
* Returns:  This is generally a longword for CON, else a byte
*           0x0000FF1A for EOF
*

cddev_getc:
 move.w   d0,-(sp)
 move.l   a0,-(sp)
 move.l   cdecl_cddev+dev_getc,a0
 jsr      (a0)
 addq.l   #6,sp
 rts


**********************************************************************
*
* long dev_getline( a0 = FD *f, a1 = char *buf, d1 = long size,
*                      d0 = int mode )
*
* mode & 0x0001:    cooked
* mode & 0x0002:    echo mode
*
* Returns:  Number of bytes read, or error code
*

cddev_getline:
 move.w   d0,-(sp)                 ; mode
 move.l   d1,-(sp)                 ; size
 move.l   a1,-(sp)                 ; buf
 move.l   a0,-(sp)                 ; FD
 move.l   cdecl_cddev+dev_getline,a0
 jsr      (a0)
 lea      14(sp),sp
 rts


**********************************************************************
*
* long dev_putc( a0 = FD *f, d0 = int mode, d1 = long value )
*
* mode & 0x0001:    cooked
*
* Returns:  Number of bytes written, 4 for a terminal
*

cddev_putc:
 move.l   d1,-(sp)                 ; val
 move.w   d0,-(sp)                 ; mode
 move.l   a0,-(sp)                 ; FD
 move.l   cdecl_cddev+dev_putc,a0
 jsr      (a0)
 lea      10(sp),sp
 rts


**********************************************************************
*
* long dev_stat(a0 = FD *f, a1 = long *unselect,
*                  d0 = int rwflag, d1 = long apcode)
*

cddev_stat:
 move.l   d1,-(sp)                 ; apcode
 move.w   d0,-(sp)                 ; rwflag
 move.l   a1,-(sp)                 ; unsel
 move.l   a0,-(sp)                 ; FD
 move.l   cdecl_cddev+dev_stat,a0
 jsr      (a0)
 lea      14(sp),sp
 rts


**********************************************************************
*
* long dev_seek(a0 = FD *f,  d0 = long where, d1 = int mode)
*

cddev_seek:
 move.w   d1,-(sp)                 ; mode
 move.l   d0,-(sp)                 ; where
 move.l   a0,-(sp)                 ; FD
 move.l   cdecl_cddev+dev_seek,a0
 jsr      (a0)
 lea      10(sp),sp
 rts


**********************************************************************
*
* long dev_ioctl(a0 = FD *f,  d0 = int cmd, a1 = void *buf)
*

cddev_ioctl:
 move.l   a1,-(sp)                 ; buf
 move.w   d0,-(sp)                 ; cmd
 move.l   a0,-(sp)                 ; FD
 move.l   cdecl_cddev+dev_ioctl,a0
 jsr      (a0)
 lea      10(sp),sp
 rts


**********************************************************************
*
* long long dev_datime(a0 = FD *file, a1 = int d[2], d0 = int set)
*

cddev_datime:
 move.w   d0,-(sp)                 ; set
 move.l   a1,-(sp)                 ; d
 move.l   a0,-(sp)                 ; FD
 move.l   cdecl_cddev+dev_datime,a0
 jsr      (a0)
 lea      10(sp),sp
 rts


**********************************************************************
*
* long long dev_close(a0 = FD *file)
*
* Writes everything back, calls the file driver and releases the
* FD if appropriate.
*

cddev_close:
 move.l   a0,-(sp)                 ; FD
 move.l   cdecl_cddev+dev_close,a0
 jsr      (a0)
 addq.l   #4,sp
 rts

     END
